 /**
* \file: Transport.h
*
* \version: 0.1
*
* \release: $Name:$
*
* Includes the interfaces to establish the transport layer to communicate
* with a mobile device via USB, BT or WiFi.
*
* \component: Unified SPI
*
* \author: D. Girnus / ADIT/SW2 / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef USPI_FEATURE_DISCOVERY_TRANSPORT_H_
#define USPI_FEATURE_DISCOVERY_TRANSPORT_H_

#include <adit_typedef.h>
#include <sys_time_adit.h>
#include <pthread_adit.h>

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string>

/* included to communicate with the devices */
#include <libusb-1.0/libusb.h>

#include "FeatureDiscoveryDefines.h"

#include <memory.h>

#include "FeatureDiscoveryTypes.h"
/* indirect included in FeatureDiscoveryTypes.h */
#include <uspi/deviceManagement.h>


namespace adit { namespace uspi {


template<typename Type> class Transport : Type
{
public:
    Transport(const DeviceInfo& inDevInfo) : Type(inDevInfo) {}

    /* open device defined by DeviceInfo and Type */
    DiscoveryError open()
    {
        return this->connect();
    }
    /* close current open device */
    DiscoveryError close(void)
    {
        return this->disconnect();
    }

    /* transmit the data to USB device which was opened */
    DiscoveryError send(uint8_t inReqType, uint8_t inReq, uint16_t inValue, uint16_t inIndex, \
                     uint8_t *DataBuf, uint16_t DataLength, uint16_t* outTransfered, int32_t inTimeout)
    {
        return this->transmit(inReqType, inReq, inValue, inIndex, DataBuf, DataLength, outTransfered, inTimeout);
    }
    /* transmit the data to BT device which was opened */
    DiscoveryError send(uint8_t *DataBuf, uint16_t DataLength, uint16_t* outTransfered, int32_t inTimeout)
    {
        return this->transmit(DataBuf, DataLength, outTransfered, inTimeout);
    }

    /* receive data from device which was opened */
    DiscoveryError recv(uint8_t *outDataBuf, uint16_t inDataLength, int32_t inTimeout)
    {
        return this->receive(outDataBuf, inDataLength, inTimeout);
    }

    /* currently, only for USB devices available */
    DiscoveryError reset(void)
    {
        return this->release();
    }
private:

};

class UsbTransport
{
public:
    UsbTransport(const DeviceInfo& inDevInfo);
    virtual ~UsbTransport();

    DiscoveryError connect(void);

    DiscoveryError disconnect(void);

    DiscoveryError transmit(uint8_t inReqType, uint8_t inReq, uint16_t inValue, uint16_t inIndex, \
                     uint8_t *DataBuf, uint16_t DataLength, uint16_t* outTransfered, int32_t inTimeout);

    DiscoveryError receive(uint8_t *outDataBuf, uint16_t inDataLength, int32_t inTimeout);

    DiscoveryError release(void);

private:
    bool isConnected(void) const { return mpLibUsbDeviceHandle ? true : false; };

    struct libusb_context* mpLibUsbContext;
    struct libusb_device_handle* mpLibUsbDeviceHandle;

    DeviceInfo mInfo;
};

class BtTransport
{
public:
    BtTransport(const DeviceInfo& inDevInfo) { mInfo = inDevInfo; }
    virtual ~BtTransport() {}

    DiscoveryError connect(void);

    DiscoveryError disconnect(void);

    DiscoveryError transmit(uint8_t *DataBuf, uint16_t DataLength, uint16_t* outTransfered, int32_t inTimeout);

    DiscoveryError receive(uint8_t *outDataBuf, uint16_t inDataLength, int32_t inTimeout);

    DiscoveryError release(void);

private:

    DeviceInfo mInfo;
};


class WiFiTransport
{
public:
    WiFiTransport(const DeviceInfo& inDevInfo) { mInfo = inDevInfo; }
    virtual ~WiFiTransport() {}

    DiscoveryError connect(void);

    DiscoveryError disconnect(void);

    DiscoveryError transmit(uint8_t *DataBuf, uint16_t DataLength, uint16_t* outTransfered, int32_t inTimeout);

    DiscoveryError receive(uint8_t *outDataBuf, uint16_t inDataLength, int32_t inTimeout);

    DiscoveryError release(void);

private:

    DeviceInfo mInfo;
};


} } /* namespace adit { namespace uspi { */


#endif /* USPI_FEATURE_DISCOVERY_TRANSPORT_H_ */
